from sage.libs.gmp.all cimport *

from sage.libs.pari.gen cimport GEN, gen, PariInstance
cdef PariInstance pari
from sage.libs.pari.gen import pari

include "../ext/stdsage.pxi"

cdef extern from "math.h":
    double mlog "log"(double)
    double mexp "exp"(double)
    double msqrt "sqrt"(double)

cdef extern from 'pari/pari.h':
    char* diffptr
    int NEXT_PRIME_VIADIFF(int, char*)
    # factor1.c
    GEN squfof(GEN)
    GEN factorint(GEN, int)

cdef extern from "ecm.h":
    ctypedef struct __ecm_param_struct:
        int method
    int ECM_ECM, ECM_PM1, ECM_PP1
    ctypedef __ecm_param_struct *ecm_params
    int ecm_factor (mpz_t p, mpz_t n, double B1, ecm_params)
    void ecm_init (ecm_params)
    void ecm_clear (ecm_params)

include "../ext/interrupt.pxi"

from integer cimport Integer
from sage.structure.factorization import Factorization

from sage.rings.integer_ring import ZZ
from sage.rings.rational_field import QQ
from sage.rings.real_double import RDF

import math, sys, os
from math import ceil
from sage.misc.all import walltime, DOT_SAGE

from sage.interfaces.qsieve import qsieve
from sage.interfaces.ecm import ecm

from sage.calculus.predefined import n as var_n, p as var_p
from sage.calculus.calculus import exp, log
from sage.gsl.integration import numerical_integral

cdef inline Integer new_Integer():
    return <Integer>PY_NEW(Integer)

cdef Integer two = Integer(2)

tuning_file = DOT_SAGE + "factor_tuning.txt"


cpdef double prob_no_prime_factor(B, int p_limit = 100000, double int_limit = mexp(25)) except -1:
    """
    Returns the probability a sufficiently large integer chosen uniformly 
    at random will have no prime divisior less than a bound B. Specifically, 
    if we let `P` be the prodcut of all primes less than B, then it estimates 
    the probabilty that an integers `n \in [0, P-1]` has `gcd(n, P) = 1`. 
    
    EXAMPLES::
    
        sage: from sage.rings.integer_factor import prob_no_prime_factor
        sage: prob_no_prime_factor(3)
        0.5
        sage: prob_no_prime_factor(10)
        0.22857142857142856
        sage: P = prod(primes(10))
        sage: sage: sum(gcd(a, P) == 1 for a in [0..P-1]) / P
        8/35
        sage: RR(8/35)
        0.228571428571429
        sage: prob_no_prime_factor(10^5)
        0.04875291785101505
        sage: prob_no_prime_factor(10^60)
        0.0040627414855629899

    Here we test our approximations::
    
        sage: prob_no_prime_factor(10^6, p_limit=10^6)
        0.040638210171648557
        sage: prob_no_prime_factor(10^6, p_limit=10^5, int_limit=10^6)
        0.040627416593501761
        sage: prob_no_prime_factor(10^6, p_limit=10^5, int_limit=10^5)
        0.040627431542512533
    """
    cdef double total = 1
    cdef long limit = p_limit
    cdef double logB, b
    if B < limit:
        limit = B
    # Compute directly for the small primes
    pari.init_primes(limit)
    cdef char *pari_prime_table = diffptr + 1
    cdef long p = 2
    while p < limit:
        total *= (p-1)/(<double>p)
        NEXT_PRIME_VIADIFF(p, pari_prime_table)
    # Compute vai a numerical integral for medium sized primes
    if B > limit:
        b = logB = mlog(B)
        if b > mlog(int_limit):
            b = mlog(int_limit)
        from sage.calculus.predefined import x
        # This integral comes from approximating sum_p log((p-1)/p) via the 
        # prime number theorem as a sum over all integers, which is approximated
        # with an integral followed by a change of variables and some algebraic manipulation. 
        total *= mexp(numerical_integral(log(1-exp(-x)) * exp(x)/x, mlog(limit), b)[0])
        # For large x, the above expression has numerical issues, however log(1-exp(-x)) * exp(x)/x ~ -1/x
        if logB > b:
            total *= b / logB # mexp(mlog(b) - mlog(logB))
            # This last approximation is quite good for large x, and gives the 
            # surprising result that a random number n has no prime divisors 
            # between a and b with probability very close to log(a)/log(b). 
    return total


def prime_product(long limit):
    """
    This computes the product of all odd primes less than limit. 
    
    EXAMPLES::
        
        sage: from sage.rings.integer_factor import prime_product
        sage: prime_product(10)
        105
        sage: factor(prime_product(100))
        3 * 5 * 7 * 11 * 13 * 17 * 19 * 23 * 29 * 31 * 37 * 41 * 43 * 47 * 53 * 59 * 61 * 67 * 71 * 73 * 79 * 83 * 89 * 97
        sage: prime_product(10^4) == prod(primes(3, 10^4))
        True
        sage: prime_product(10^6).nbits()
        1440508
    """
    # We use a stack here to compute a balanced product, which allows the bulk
    # of the work to be asymptotically fast multiplication of similarly sized
    # integers and makes a huge difference in runtime.
    cdef list stack = []
    cdef Integer z
    pari.init_primes(limit)
    cdef char *pari_prime_table = diffptr + 2
    cdef long p = 3
    cdef long cc, c = 0
    cdef long j
    while p < limit:
        z = PY_NEW(Integer)
        mpz_set_ui(z.value, p)
        NEXT_PRIME_VIADIFF(p, pari_prime_table)
        # The p fit comfortably in a word, and their product is well below
        # any of the asymptotically fast algorithms for a while, and we can
        # save on the product balancing overhead.
        # The timing curve seems pretty flat around 20-200, so we just hard 
        # code a value. 
        for j from 0 <= j < 150:
            if p >= limit:
                break
            mpz_mul_ui(z.value, z.value, p)
            NEXT_PRIME_VIADIFF(p, pari_prime_table)
        stack.append(z)
        c += 1
        # The binary expansion of c tells us how many times to "fold" the 
        # top result in. 
        cc = c
        while cc & 1 == 0:
            t = stack.pop()
            stack[-1] *= t
            cc >>= 1
    while len(stack) > 1:
        t = stack.pop()
        stack[-1] *= t
    return stack[-1]


def L(a, c, n=var_n):
    return exp(c*log(n)**a*log(log(n))**(1-a))

cdef int digits(n) except -1:
    if n == 0:
        return 1
    else:
        return ceil(mlog(n)/mlog(10))




cpdef factor_main(Integer n, smallest_possible_prime=2, int verbosity=0, bint special_forms=False):
    """
    EXAMPLES::
    
        sage: from sage.rings.integer_factor import factor_main
        sage: factor_main(6)
        [(2, 1), (3, 1)]
    
        sage: factor_main(12, verbosity=2)
        Calling TrialDivisionGCD(10000)
        Done. Trivial cofactor after trial division.
        [(2, 2), (3, 1)]
        
        sage: factor_main(12 * next_prime(10^5), verbosity=2)
        Calling TrialDivisionGCD(100000)
        Done. Prime cofactor after trial division.
        [(2, 2), (3, 1), (100003, 1)]
        
        sage: factor_main(12 * next_prime(10^5)^3, verbosity=2)
        Calling TrialDivisionGCD(100000)
        Perfect power discovered: 3
        Done. Prime cofactor.
        [(2, 2), (3, 1), (100003, 3)]

        sage: factor_main(next_prime(10^10) * next_prime(11^12), verbosity=2)
        Calling TrialDivisionGCD(100000)
        Calling PollardPPlus1(15)
        Done. Prime cofactor.
        Done. Prime cofactor.
        [(10000000019, 1), (3138428376749, 1)]
        
        sage: factor_main(next_prime(13^15) * next_prime(17^12)^3, verbosity=2) # random
        Calling TrialDivisionGCD(100000)
        Calling EllipticCurveMethod(15)
        Perfect power (3) discovered.
        Done. Prime cofactor.
        Done. Prime cofactor.
        [(582622237229771, 3), (51185893014090761, 1)]
    """
    
    cdef double time_taken, time_savings, total_savings, prob_success
    cdef double best_savings, original_time
    cdef double logn = mlog(n)
    cdef double digits = logn/mlog(10)
    cdef int i

    # First we trial divide...
    best = None
    if smallest_possible_prime < 1e5:
        for t in trial_division_methods:
            if t.limit > smallest_possible_prime:
                digit_savings = t.expected_savings(n, smallest_possible_prime)
                time_taken = t.est_time_digits(digits)
                time_savings = sieve_time(digits) - sieve_time(digits-digit_savings)
                if verbosity > 2:
                    print t, "takes %.3g saves %.3g seconds" % (time_taken, time_savings)
                if best is None or time_savings - time_taken > best_savings:
                    best = t
                    best_savings = time_savings - time_taken
                    
    if best is not None:
        if verbosity > 0:
            print "Calling", best
        factors = best(n)
        smallest_possible_prime = best.limit
        cofactor = factors[-1][0]
        if cofactor < best.limit:
            if verbosity > 1:
                print "Done. Trivial cofactor after trial division."
            return factors
        elif len(factors) > 1:
            return factors[:-1] + factor_main(cofactor, smallest_possible_prime, verbosity)
            
    
    # Now we need to filter out primes and powers
    if n.is_prime():
        if verbosity > 1:
            print "Done. Prime cofactor."
        return [(n, 1)]
    
    cdef int max_power = <int>(logn / mlog(smallest_possible_prime) + 0.5)
    if max_power < 3 and n.is_square() or n.is_perfect_power():
        if max_power < 3 or n.is_square():
            root = n.sqrt()
            power = 2
        else:
            for i from 3 <= i <= max_power:
                root, is_root = n.nth_root(i, True)
                if is_root:
                    power = i
                    break
        if verbosity > 1:
            print "Perfect power (%s) discovered." % power
        factors = factor_main(root, smallest_possible_prime, verbosity)
        return [(p, e*power) for p, e in factors]

    # Perhpas our input is rather special.
    # We won't waste too much time here, but it might be worth checking once...
    if special_forms:
        for m in special_prime_methods:
            factors = m(n)
            if len(factors) > 1:
                return recursive_factor(factors, smallest_possible_prime, verbosity)
    
    while True:
        best = None
        original_time = sieve_time(digits)
        best_savings = 0
        for t in probabilistic_methods:
            prob_success = t.prob_success(n, smallest_possible_prime)
            digit_savings = t.expected_savings(n, smallest_possible_prime)
            time_taken = t.est_time_digits(digits)
            time_savings = original_time - sieve_time(digits-digit_savings)
            total_savings = prob_success * (time_savings - time_taken)
            if verbosity > 2:
                print t, "takes %.3g saves %.3g seconds with probability %.2g" % (time_taken, time_savings, prob_success)
            if total_savings > best_savings:
                best = t
                best_savings = total_savings
        if best is None:
            break
        else:
            if verbosity > 0:
                print "Calling", best
            factors = best(n)
            if len(factors) > 1:
                return recursive_factor(factors, smallest_possible_prime, verbosity)
            smallest_possible_prime = best.limit # should I get it all the way up to here?
    
    best = None
    for t in generic_methods:
        time = t.est_time_digits(digits)
        if best is None or time < best_time:
            best = t
            best_time = time
    
    if verbosity > 0:
        print "Calling", best

    return recursive_factor(best(n), smallest_possible_prime, verbosity)


cpdef sieve_time(d):
    best = None
    cdef double best_time
    for m in generic_methods:
        time = m.est_time_digits(d)
        if best is None or time < best_time:
            best = m
            best_time = time
    return best_time

cpdef recursive_factor(factors, smallest_possible_prime, verbosity):
    all = []
    for pp, pe in factors:
        sub_factors = factor_main(pp, smallest_possible_prime, verbosity)
        all += [(p, pe*e) for p, e in sub_factors]
    return consolidate_factors(all)

cpdef consolidate_factors(all):
    all.sort()
    consolidated = []
    last_p = None
    for p, e in all:
        if p == last_p:
            consolidated[-1] = p, consolidated[-1][1] + e
        else:
            consolidated.append((p,e))
            last_p = p
    return consolidated
            


cdef class FactorizationMethod:

    cdef public list _tuning_timings
    cdef public list _tuning_coeffs

    def __call__(self, n):
        raise NotImplementedError
    
    def __cmp__(self, other):
        if isinstance(self, FactorizationMethod) and isinstance(other, FactorizationMethod):
            return cmp(repr(self), repr(other))
        else:
            return cmp(type(self), type(other))
    
    def __repr__(self):
        return self.__class__.__name__
    
    def time(self, n, min_time=0.1):
        cdef int i, count
        t = walltime()
        self(n)
        t = walltime(t)
        if min_time and t < min_time:
            count = ceil(min_time/t)
            while t < min_time:
                count = ceil(1.5 * count)
                t = walltime()
                for i in range(count):
                    self(n)
                t = walltime(t)
            t /= count
        return t

    def timings(self, ns, time_limit=False, min_time=0.1, verbose=False):
        all = []
        start = walltime()
        for n in ns:
            if verbose:
                print "%r %s digits..." % (self, digits(n)), 
            t = self.time(n, min_time)
            print "%.3g seconds" % t
            all.append((n,t))
            if time_limit and walltime(start) > time_limit and len(all) >= len(self.big_O_list()):
                break
        return all
    
    def tune(self, ns=None, timings=None, time_limit=False, verbose=False):

        cdef double c
        cdef int i
        if timings is None:
            if isinstance(ns, int):
                nsamples = ns
                ns = None
            else:
                nsamples = 10
            if ns is None:
                a = self.min_digits()
                b = self.max_digits()
                c = float(b-a)/(nsamples-1)
                ns = []
                for i in range(nsamples):
                    s = int(10**(a/2))
                    p = (s+ZZ.random_element(s)).next_probable_prime()
                    q = (2*s+ZZ.random_element(s)).next_probable_prime()
                    if digits(p*q) > b:
                        break
                    ns.append(p*q)
                    a += c
            timings = self.timings(ns, time_limit, verbose=verbose)
            
        asym = [self.big_O_list(RDF(n)) for n,t in timings]
        from scipy import linalg
        self._tuning_timings = timings
        self._tuning_coeffs = [float(a) for a in linalg.lstsq(asym, zip(*timings)[1])[0]]
        for i, a in enumerate(self._tuning_coeffs):
            if a < 0:
                # print i, a, self._tuning_coeffs
                self._tuning_coeffs[i] = 0
    
    def est_time(self, n):
        if self._tuning_coeffs is None:
            return 1e100
        n = RDF(n)
        xs = self.big_O_list(n)
        return max(0, sum([a*x for a,x in zip(self._tuning_coeffs, xs)]))
    
    def est_time_digits(self, d):
        return self.est_time(10.0**d)
    
    def big_O_list(self, n=var_n):
        return [1, log(n), log(n)**2, self.big_O(n)]

    def big_O(self, n=var_n):
        return L(QQ(1)/2, 1, n)
    
    def min_digits(self):
        return 10
    
    def max_digits(self):
        return 80
    
    def split_factor(self, f, n):
        return [(f, 1), (n.divide_knowing_divisible_by(f), 1)]
    
    def plot(self, rgbcolor=None):
        """
        EXAMPLES::
        
            sage: from sage.rings.integer_factor import all_methods, generic_methods
            sage: P = sum(a.plot() for a in all_methods if a._tuning_coeffs is not None)
            sage: P.show(ymax=250, xmax=100, figsize=10)
            sage: P.show(ymax=50, xmax=100, figsize=10)
            sage: P.show(ymax=0.001, xmax=250, figsize=10)
            sage: P = sum(a.plot() for a in generic_methods if a._tuning_coeffs is not None)
            sage: P.show(ymax=250, xmax=100, figsize=10)
        """
        if self._tuning_coeffs is None:
            raise ValueError, "No timing data available."
        if rgbcolor is None:
            rgbcolor = self.default_rgbcolor()
        from sage.plot.all import plot, point
        G = plot(self.est_time_digits, (self.min_digits(), 1.25*self.max_digits()), rgbcolor=rgbcolor)
        if self._tuning_timings is not None:
            for (n, t) in self._tuning_timings:
                G += point((mlog(n)/mlog(10), t), rgbcolor=rgbcolor)
        return G

    def default_rgbcolor(self):
        return (0, 0, 0)


cdef class SpecialPrimeFactorizationMethod(FactorizationMethod):
    
    cdef readonly limit
    
    def __init__(self, limit):
        self.limit = limit

    def __repr__(self):
        return "%s(%s)" % (self.__class__.__name__, self.limit)

    def big_O(self, n=var_n):
        return log(n) # fixed limit only impacts constant factor
    
    def big_O_list(self, n=var_n):
        return [1, self.big_O(n)]

    def expected_savings(self, n=None, smallest_possible_prime=2):
        """
        Returns the expected number of decimal digits of factors found by 
        this method on success.
        
        EXAMPLES::
        """
        # Be optimistic, there are more big primes than small
        return mlog(self.limit) / mlog(10)

    cpdef prob_possible_prime(self, n, smallest_possible_prime=2):
        B = min(float(self.limit), msqrt(n))
        return 1 - prob_no_prime_factor(B) / prob_no_prime_factor(smallest_possible_prime)


cdef class TrialDivision(SpecialPrimeFactorizationMethod):

    def __call__(self, Integer n):
        cdef Integer cofactor = new_Integer()
        mpz_init_set(cofactor.value, n.value)
        return self.trial_divide_from(cofactor, 2, self.limit, 0, NULL)
    
    def expected_savings(self, n=None, smallest_possible_prime=2):
        """
        Returns the expected number of decimal digits of factors found by 
        this method. 
        
        EXAMPLES::

            sage: from sage.rings.integer_factor import TrialDivision
            sage: TrialDivision(10^4).expected_savings()
            3.7549668884566065
            sage: TrialDivision(10^5).expected_savings()
            4.7510303158877401
            sage: TrialDivision(10^5).expected_savings(smallest_possible_prime=10^4)
            0.99606342743115228

        """
        cdef double total = 0
        cdef int min_prime, p = 0
        pari.init_primes(self.limit)
        cdef char* pari_prime_table = diffptr
        if smallest_possible_prime < self.limit:
            min_prime = smallest_possible_prime
            while p < min_prime:
                NEXT_PRIME_VIADIFF(p, pari_prime_table)
            while p < self.limit:
                total += mlog(p) / (p-1) # sum_k=0^oo k(p-1)log(p)/p^(k+1)
                NEXT_PRIME_VIADIFF(p, pari_prime_table)
            return total / mlog(10)
        else:
            return 0
    
    def prob_success(self, n, smallest_possible_prime=2):
        # If it's there, we'll find it. 
        return self.prob_possible_prime(n, smallest_possible_prime)

    cdef trial_divide_from(self, Integer cofactor, int start, int limit, int p, char* pari_prime_table):
        cdef long e
        cdef Integer z
        factors = []
        if start <= 2:
            if mpz_even_p(cofactor.value):
                e = mpz_scan1(cofactor.value, 0)
                mpz_fdiv_q_2exp(cofactor.value, cofactor.value, e)
                factors.append((two, e))
                if mpz_cmp_ui(cofactor.value, 1) == 0:
                    return factors
        if pari_prime_table == NULL:
            p = 3
            pari_prime_table = diffptr + 2
        _sig_on
        while p < start:
            NEXT_PRIME_VIADIFF(p, pari_prime_table)
        while p < limit:
            e = 0
            while mpz_divisible_ui_p(cofactor.value, p):
                # We repeat the division here, assuming that 
                # mpz_divisible_ui_p is faster and usually false.
                mpz_divexact_ui(cofactor.value, cofactor.value, p)
                e += 1
            if e != 0:
                z = <Integer>PY_NEW(Integer)
                mpz_set_ui(z.value, p)
                factors.append((z, e))
                if mpz_cmp_ui(cofactor.value, 1) == 0:
                    break
            NEXT_PRIME_VIADIFF(p, pari_prime_table)
        _sig_off
        if mpz_cmp_ui(cofactor.value, 1) != 0:
            factors.append((cofactor, 1))
        return factors
    
    def default_rgbcolor(self):
        return (0, 0, mlog(self.limit) / mlog(10.0) / 7)
        
    def max_digits(self):
        return 250


class TrialDivisionPacked(TrialDivision):

    def __call__(TrialDivision self, Integer n):
    
        cdef int limit = self.limit
    
        cdef list factors = []
        cdef unsigned long p, pp
        cdef long e
        cdef Integer z
        cdef bint done = False
        
        cdef Integer cofactor = PY_NEW(Integer)
        mpz_set(cofactor.value, n.value)
        if mpz_even_p(cofactor.value):
            e = mpz_scan1(cofactor.value, 0)
            mpz_fdiv_q_2exp(cofactor.value, cofactor.value, e)
            z = <Integer>PY_NEW(Integer)
            mpz_set_ui(z.value, 2)
            factors.append((z, e))
            done = mpz_cmp_ui(cofactor.value, 1) == 0 or cofactor.is_prime()
        
        _sig_on
        cdef unsigned long prod
        cdef unsigned long prod_primes[17]
        cdef unsigned long r
        cdef int i, prod_len, res
        cdef bint found_factor = False
        
        # Might as well use Pari's table of primes instead of making our own.
        cdef char* pari_prime_table
        
        pari.init_primes(limit)
        pari_prime_table = diffptr + 2
        p = 3
        prod_len = 0
        while p < limit and prod_len != 1:
            prod_len = 1
            prod = prod_primes[0] = p
            NEXT_PRIME_VIADIFF(p, pari_prime_table)
            while p < limit and (<unsigned long>-1) / prod > p:
                prod *= p
                prod_primes[prod_len] = p
                prod_len += 1
                NEXT_PRIME_VIADIFF(p, pari_prime_table)
            prod_primes[prod_len] = p # need to recover this later
            found_factor = False
            r = mpz_fdiv_ui(cofactor.value, prod)
            for i from 0 <= i < prod_len:
                p = prod_primes[i]
                if r % p == 0:
                    found_factor = True
                    e = 1
                    mpz_divexact_ui(cofactor.value, cofactor.value, p)
                    while mpz_divisible_ui_p(cofactor.value, p):
                        # We repeat the division here, assuming that 
                        # mpz_divisible_ui_p is faster and usually false.
                        mpz_divexact_ui(cofactor.value, cofactor.value, p)
                        e += 1
                    z = <Integer>PY_NEW(Integer)
                    mpz_set_ui(z.value, p)
                    factors.append((z, e))
            p = prod_primes[prod_len]
            if found_factor and mpz_cmp_ui(cofactor.value, 1) == 0:
                done = True
                break
                
        _sig_off

        if not done:
            factors += self.trial_divide_from(cofactor, p, limit, p, pari_prime_table)
        return factors

    def default_rgbcolor(self):
        return (0, .325, mlog(self.limit) / mlog(10.0) / 7)



cdef class TrialDivisionGCD(TrialDivision):

    # We store the product of all odd primes up to prime_prod_limit, useful for the 
    # trial divison stage of factoring. For prime_prod_limit = 10^5, this is only 
    # a 17.5 KB number and should take < 10ms to compute. 
    cdef readonly int prime_prod_limit
    cdef readonly Integer prime_prod_cache
    
    def __init__(self, limit, gcd_limit=None):
        """
            sage: from sage.rings.integer_factor import TrialDivisionGCD
        """
        TrialDivision.__init__(self, limit)
        if gcd_limit is None:
            gcd_limit = limit
        self.prime_prod_limit = gcd_limit
        self.prime_prod_cache = prime_product(limit)

    def __call__(TrialDivision self, Integer n):
    
        cdef int limit = self.limit # may overflow on unreasonable values
    
        cdef list factors = []
        cdef long e
        cdef Integer z
        cdef bint done = False
        
        cdef Integer cofactor = PY_NEW(Integer)
        mpz_init_set(cofactor.value, n.value)
        
        if mpz_even_p(cofactor.value):
            e = mpz_scan1(cofactor.value, 0)
            mpz_fdiv_q_2exp(cofactor.value, cofactor.value, e)
            factors.append((two, e))
            done = mpz_cmp_ui(cofactor.value, 1) == 0
        
        cdef Integer g = self.prime_prod_cache.gcd(n)
        cdef unsigned long g_long = 0
        if mpz_fits_ulong_p(g.value):
            g_long = mpz_get_ui(g.value)

        cdef long min_limit = limit if limit < self.prime_prod_limit else self.prime_prod_limit
        # Might as well use Pari's table of primes instead of making our own.
        cdef char* pari_prime_table = diffptr + 2
        cdef unsigned long p = 3
        
        _sig_on
        while p < min_limit and g_long != 1:
            if (g_long and g_long % p == 0) or mpz_divisible_ui_p(g.value, p):
                e = 1
                mpz_divexact_ui(cofactor.value, cofactor.value, p)
                while mpz_divisible_ui_p(cofactor.value, p):
                    # We repeat the division here, assuming that 
                    # mpz_divisible_ui_p is faster and usually false.
                    mpz_divexact_ui(cofactor.value, cofactor.value, p)
                    e += 1
                z = <Integer>PY_NEW(Integer)
                mpz_set_ui(z.value, p)
                factors.append((z, e))
                if g_long:
                    g_long /= p
                else:
                    mpz_divexact_ui(g.value, g.value, p)
                    if mpz_fits_ulong_p(g.value):
                        g_long = mpz_get_ui(g.value)
            NEXT_PRIME_VIADIFF(p, pari_prime_table)
        _sig_off
            
        if mpz_cmp_ui(cofactor.value, 1) != 0:
            if self.prime_prod_limit < self.limit:
                factors += self.trial_divide_from(cofactor, self.prime_prod_limit, limit, p, pari_prime_table)
            else:
                factors.append((cofactor, 1))
        return factors

    def default_rgbcolor(self):
        return (0, .75, mlog(self.limit) / mlog(10.0) / 7)



class FermatMethod(SpecialPrimeFactorizationMethod):
    
    def __call__(self, Integer n):
        """
        This uses Fermant's method of factorization, which is good 
        for finding prime factors close to sqrt(n), but an exponentially
        slow method in general. 
        
        EXAMPLES::
        
            sage: from sage.rings.integer_factor import FermatMethod
            sage: FermatMethod(10)(next_prime(10^5) * previous_prime(10^5))
            [(99991, 1), (100003, 1)]
            sage: n = next_prime(10^50) * previous_prime(10^50)
            sage: FermatMethod(10)(n)
            [(99999999999999999999999999999999999999999999999943, 1), (100000000000000000000000000000000000000000000000151, 1)]
            sage: n = next_prime(10^20) * next_prime(10^20 + 10^10)
            sage: FermatMethod(10)(n)
            [(100000000000000000039, 1), (100000000010000000069, 1)]
            sage: n = next_prime(10^20) * next_prime(2*10^20)
            sage: FermatMethod(10^5)(n)
            [(20000000000000000016700000000000000003471, 1)]
        """
        # Close prime, not small prime
        cdef int i
        cdef Integer A = new_Integer()
        cdef Integer B = new_Integer()
        cdef Integer twoA = new_Integer()

        # A = ceil(sqrt(n))
        mpz_sqrt(A.value, n.value)
        mpz_add_ui(A.value, A.value, 1)
        # twoA = 2*A
        mpz_mul_2exp(twoA.value, A.value, 1)
        # B = A^2 mod n
        mpz_mul(B.value, A.value, A.value)
        mpz_sub(B.value, B.value, n.value)
        _sig_on
        for i from 0 <= i < self.limit:
            if mpz_perfect_square_p(B.value):
                return self.split_factor(A + i - B.sqrt(), n)
            mpz_add(B.value, B.value, twoA.value)
            mpz_add_ui(B.value, B.value, 1)
        _sig_off
        return [(n, 1)]


class OneLineFactorization(SpecialPrimeFactorizationMethod):
    
    def __call__(self, Integer n):
        """
        Bill Hart's "One line factorization" algorithm. This is good for 
        finding products of factors of the form (a^b + c). 
        
        EXAMPLES::
            
            sage: from sage.rings.integer_factor import OneLineFactorization
            sage: OneLineFactorization(100)(next_prime(10^20) * next_prime(10^21))
            [(1000000000000000000117, 1), (100000000000000000039, 1)]
            sage: OneLineFactorization(10000)(next_prime(3^50) * next_prime(3^55-1000))
            [(174449211009120179071169621, 1), (717897987691852588770277, 1)]
        """
        # Not really a small prime method, but only goes up to a certain limit
        cdef int i
        cdef Integer s = new_Integer()
        cdef Integer m = new_Integer()
        cdef Integer Mni = new_Integer()
        from sage.rings.integer_mod import mod
        for i from 1 <= i < self.limit:
            mpz_mul_ui(Mni.value, n.value, 480*i)
            mpz_sqrt(s.value, Mni.value)
            mpz_add_ui(s.value, s.value, 1) # want ceil got floor
            mpz_mul(m.value, s.value, s.value)
            mpz_sub(m.value, m.value, Mni.value)
            mpz_fdiv_r(m.value, m.value, n.value)
            if mpz_perfect_square_p(m.value):
                f = n.gcd(s - m.sqrt())
                if f != 1 and f != n:
                    return self.split_factor(f, n)
        return [(n, 1)]


cdef class SmoothOrderMethod(SpecialPrimeFactorizationMethod):
    
    cdef readonly int digits
    
    def __init__(self, int digits):
        SpecialPrimeFactorizationMethod.__init__(self, ZZ(10)**digits)
        self.digits = digits
    
    def __repr__(self):
        return "%s(%s)" % (self.__class__.__name__, self.digits)
    
    def min_digits(self):
        return 2*self.digits


class EllipticCurveMethod(SmoothOrderMethod):

    def __call__(self, n):
        ff = ecm.find_factor(n, factor_digits=self.digits, c=ecm.recommended_ncurves(self.digits))
        return [(f, 1) for f in ff]
    
    def time(self, n, min_time):
        c = 3
        t = 0
        while t < min_time and c < 1000000:
            c *= 2
            t = walltime()
            ecm.find_factor(n, factor_digits=self.digits, c=c)
            t = walltime(t)
        return t / c * ecm.recommended_ncurves(self.digits)
        
    def prob_success(self, n, smallest_possible_prime=2):
        return (1-math.exp(-1)) * self.prob_possible_prime(n, smallest_possible_prime)

#    def big_O(self, n=var_n):
#        return log(ecm.recommended_B1(self.digits)) * ecm.recommended_ncurves(self.digits) * log(n)

    def default_rgbcolor(self):
        return (1, self.digits/70.0, 0)



cdef class PollardPMinus1(SmoothOrderMethod):
    
    def __call__(self, Integer n):
        B1 = ecm.recommended_B1(self.digits)
        cdef Integer f = new_Integer()
        cdef ecm_params params
        ecm_init(params)
        # See README of gmp-ecm
        if isinstance(self, PollardPPlus1):
            B1 *= 5
            params.method = ECM_PP1
        else:
            B1 *= 10
            params.method = ECM_PM1
        _sig_on
        cdef int res = ecm_factor(f.value, n.value, B1, params)
        _sig_off
        ecm_clear(params)
        if res < 0:
            raise RuntimeError, "Error running GMP-ECM"
        elif res == 0:
            return [(n, 1)]
        else:
            return self.split_factor(f, n)

    def prob_success(self, n, smallest_possible_prime=2):
        B1 = ecm.recommended_B1(self.digits) * 5
        alpha = log(RDF(self.limit)) / log(RDF(B1))
        p_smooth = alpha ** -alpha # rough approx
        return p_smooth * self.prob_possible_prime(n, smallest_possible_prime)

    def default_rgbcolor(self):
        return (1, self.digits/70.0, (1+isinstance(self, PollardPPlus1))/3)

#    def big_O(self, n=var_n):
#        factor_digits = digits(self.limit)
#        B1 = ecm.recommended_B1(factor_digits)
#        return B1 * log(B1) * log(n)    


cdef class PollardPPlus1(PollardPMinus1):
    pass


class QuadraticSieve(FactorizationMethod):
    
    def __call__(self, n):
        factors = qsieve(n)[0]
        # qsieve returns all factors (and cofactors) found
        decomposition = factors[:2]
        for a in factors[2:]:
            if all([a % p != 0 for p in decomposition]):
                decomposition.append(a)
        return [(p, 1) for p in decomposition]
        
    def big_O(self, n=var_n):
        return L(QQ(1)/2, 1, n)
    
    def min_digits(self):
        return 40

    def default_rgbcolor(self):
        return (0, .5, 0)

class PariFactor(FactorizationMethod):
    
    def __call__(self, Integer n):
        cdef gen pari_n = pari.new_gen_from_mpz_t(n.value)
        _sig_on
        return self.pari_factorization_helper(pari.new_gen(factorint(pari_n.g, 0)))
    
    def pari_factorization_helper(self, pari_res):
        pari_factors = list(pari_res[0])
        pari_exponents = list(pari_res[1])
        factors = []
        cdef long e
        for p, e in zip(pari_factors, pari_exponents):
            factors.append((Integer(p), e))
        return factors
        
    def max_digits(self):
        return 65

    def default_rgbcolor(self):
        return (0, .5, .5)

class PariQS(PariFactor):
    
    def __call__(self, Integer n):
        cdef gen pari_n = pari.new_gen_from_mpz_t(n.value)
        _sig_on
        return self.pari_factorization_helper(pari.new_gen(factorint(pari_n.g, 0x6)))

    def default_rgbcolor(self):
        return (0, .5, 0.25)


class SquareForms(FactorizationMethod): # Also known as SQUFOF

    def __call__(self, Integer n):
        cdef gen pari_n = pari.new_gen_from_mpz_t(n.value)
        _sig_on
        z = Integer(pari.new_gen(squfof(pari_n.g)))
        return [(z, 1), (n//z, 1)]
    
    def big_O(self, n=var_n):
        return n ** (QQ(1)/4)

    def min_digits(self):
        return 10
    
    def max_digits(self):
        return digits(ZZ(2)**59)

    def default_rgbcolor(self):
        return (1, 0, 1)




trial_division_methods = [TrialDivision(10000), TrialDivision(100000), 
                          TrialDivisionPacked(10000), TrialDivisionPacked(100000),
                          TrialDivisionGCD(10000), TrialDivisionGCD(100000)]

special_prime_methods = [FermatMethod(1000), OneLineFactorization(10000)]

probabilistic_methods = ([EllipticCurveMethod(n) for n in range(15, 50, 5)] +
                            [PollardPMinus1(n) for n in range(15, 50, 10)] + 
                            [PollardPPlus1(n) for n in range(15, 50, 10)])

generic_methods = [PariFactor(), PariQS(), QuadraticSieve()] # PariFactor uses squareforms

all_methods = trial_division_methods + special_prime_methods + probabilistic_methods + generic_methods

def filter_methods(s):
    """
    EXAMPLES::
    
        sage: from sage.rings.integer_factor import filter_methods
    """
    s = s.lower().replace("ecm", "ellipticcurve")
    found = []
    for m in all_methods:
        for ss in s.split(','):
            if ss in repr(m).lower():
                found.append(m)
                break
    return found
    

def tune_all(time_limit=60, force=False):
    """
    EXAMPLES::
        
        sage: from sage.rings.integer_factor import tune_all
        sage: tune_all(120)

        sage: from sage.rings.integer_factor import trial_division_methods
        sage: m = trial_division_methods[0]
        sage: m._tuning_timings
    """
    if force:
        all = {}
    else:
        all = load_tuning_data()
    for method in all_methods:
        method_str = repr(method)
        if method_str not in all or force:
            print method
            try:
                method.tune(time_limit=time_limit, verbose=True)
                all[method_str] = method._tuning_timings # [(mlog(n)/mlog(10), t) for n, t in method._tuning_timings]
            except Exception, ex:
                print ex
                pass
    f = open(tuning_file, 'w')
    for method, data in all.items():
        f.write("%s: %s\n" % (str(method), "\t".join(["(%.3g, %.5g)" % (mlog(n)/mlog(10), t) for n, t in data])))
    f.close()

def load_tuning_data():
    all = {}
    if os.path.exists(tuning_file):
        for line in open(tuning_file).readlines():
            line = line.strip()
            if line and line[0] != '#':
                try:
                    method, data_str = line.split(':')
                    method = method.strip()
                    data = []
                    for datum in data_str.split('\t'):
                        digits, time = datum.strip()[1:-1].split(',')
                        data.append((10.0**float(digits), float(time)))
                    all[method] = data
                except Exception, ex:
                    print "Error parsing data:", line, ex
    for method in all_methods:
        method_str = repr(method)
        if method_str in all:
            method.tune(timings=all[method_str])
    return all

load_tuning_data()
