###############################################################################
#
#   SAGE: System for Algebra and Geometry Experimentation
#
#       Copyright (C) 2009 Dustin Moody <dbm25@math.washington.edu>
#
#  Distributed under the terms of the GNU General Public License (GPL)
#
#    This code is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
#
#  The full text of the GPL is available at:
#
#                  http://www.gnu.org/licenses/
###############################################################################




def cfactor(N,k=1,L=80,B=65):
	"""
	Continued fraction factoring algorithm, as described in M. A. Morrison and J. Brillhart, A method of factoring and the factorization of F_7, Math. Comp. 29, 1975.  (183-205)
	We first expand vN, or vkN for some suitably chosen k=1, into a simple continued fraction vkN=[q_0,q_1,q_2,,q_(n-1),v(kN+P_n )/Q_n ].  
	If A_n/B_n  is the Nth convergent, then it is a well-known identity that A_(n-1)^2-kNB_(n-1)^2=(-1)^n Q_n
	and so A_(n-1)^2=(-1)^n Q_n mod N.  We then look for a subset Q_i of {Q_1,Q_2,,Q_n} such that ?_i?(-1)^i Q_i ? is a square.  If this is not possible, we expand vkN further.  
	This yields the congruence A^2=?_i?A_(i-1)^2=?_i?(-1)^i Q_i=Q^2 ?? mod N.
	We then compute D = gcd(A-Q,N), and if 1<D<N, we have found a nontrivial factor of N
	Note--this program is not very fast.
	
	The parameter L is the bound for how far we compute the continued fraction expansion of vkN=[q_0,q_1,q_2,,q_(L-1),v(kN+P_n )/Q_L].
	The parameter B is an upper bound for the primes in the factor base.  	
	
	Example:
	sage: p1=next_prime(10^6+500)
	sage: p2=next_prime(10^5+500)
	sage: cfactor(p1*p2,B=1000,L=1600)
	100501

	"""

	g=floor(sqrt(k*N))
	global setFB
	
	# Initialize variables in continued fraction expansion
	A=[]
	Q=[]
	r=[]
	P=[]
	q=[]
	A.append(0)
	A.append(1)
	Q.append(0)
	Q.append(k*N)
	r.append(0)
	r.append(g)
	P.append(0)
	P.append(0)
	P.append(0)
	Q.append(1)
	q.append(0)
	q.append(0)

	#Find continued fraction of sqrt(k*N)
	for i in range(0,L+1):
		q.append(floor((g+P[i+2])/Q[i+2]))
		r.append(g+P[i+2]-q[i+2]*Q[i+2])
		A.append((q[i+2]*A[i+1]+A[i]) % N)
		P.append(g-r[i+2])
		Q.append(Q[i+1]+q[i+2]*(r[i+2]-r[i+1]))
		# Check if expansion is periodic too quickly.
		if Q.count(Q[len(Q)-1])>3:
                    print("Try a different k")
                    return
		# Check if a square is found in expansion without needing to find relations.
		if is_square(Q[i+3]):
			if (i % 2)==1 and gcd(A[i+2]-sqrt(Q[i+3]),N)>1 and gcd(A[i+2]-sqrt(Q[i+3]),N)<N:
				return -1,gcd(A[i+2]-sqrt(Q[i+3]),N)
	
	#Build factor base
	FB=[2]
	Pr=primes(3,B)
	for p in Pr:
		ls=legendre_symbol(N,p)
		if ls==0:
			return p
			break
		if ls==1:
			FB.append(p)
	setFB=Set(FB)

	
	#Find FB smooth Q[i]
	F=[]
	G=[]
	for i in range(3,len(Q)-1):
		if issmooth(Q[i]):
			F.append((-1)^i*Q[i])
			G.append(i)

	V=VectorSpace(GF(2),len(FB)+1+len(F))
	f=[]
	for i in range(0,len(F)):
		e=[]
		e.append((1-sgn(F[i]))/2)
		for j in range(0,len(FB)):
			e.append(F[i].ord(FB[j]) % 2)
		for j in range(0,len(F)):
			if j==i:
				e.append(1)
			else:
				e.append(0)
		f.append(V(e))

	#Find relations which produce a square mod N
	rr=len(FB)
	while rr>0:
		flg=0
		for i in range(len(F)):
			if f[i][rr]==1 and flg>0:
				f[i]=f[i]+f[flg]
			if f[i][rr]==1 and flg==0:
				flg=i
		rr=rr-1

	for ff in f:
		mx=ff.support()[0]
		if mx>(len(FB)+1):
			a=1
			qq=1
			for w in ff.support():
				a=a*A[Q.index(abs(F[w-len(FB)-1]))-1] % N
				qq=qq*F[w-len(FB)-1]
			qs=abs(qq).sqrt()
			g1=gcd(a+qs,N)
			if g1>1 and g1<N:
				return g1
			g2=gcd(a-qs,N)
			if g2>1 and g2<N:
				return g2
	print("Try again")
	return


def issmooth(d):
		"""
		Checks if a prime d is FB-smooth
		"""
		dp=Set(d.prime_factors())
		if dp.intersection(setFB)==dp:
			return True
		else:
			return False

def f(p,k,N):
	"""	
	Auxiliary function to try and determine a good value of k
	"""
	if p==2:
		if k%2==0:
			return 1/3
		if (k*N)%4==3:
			return 1/3
		if (k*N)%8==5:
			return 2/3
		if (k*N)%8==1:
			return 4/3
	if (k%p)==0:
		return 1/(p+1)
	return (2*p)/(p^2-1)

def F(k,N):
	"""
	Auxiliary function to determine a good value of k.  See Brillhart and Morrison's paper for details.
	"""
	tot=0
	for p in Pr:
		tot=tot+f(p,k,N)
	return (tot-(log(k))/2).n()

def factorbase(Bd):
	"""
	Builds factor base for use in determining a good value of k.
	"""
	global Pr
	Pr=[]
	for p in primes(1,Bd):
		Pr.append(p)


	