from sage.all import FreeModule, NumberFieldElement
from sage.all import RealField, ComplexField, Rational, RDF

def left_eigenvectors(M, field='real', prec=80):
    if field == 'real':
        field = RealField(prec=prec)
    elif field == 'complex':
        field = ComplexField(prec=prec)
    fcp = M.fcp()
    evecs = []
    eigenvals = []
    if len(fcp) == 1: # Irreducible case
        roots = [r[0] for r in fcp[0][0].roots(field)]
        ev = ws_ev(M, field)
        for r in roots:
            v = [x.polynomial()(r) for x in ev]
            evecs.append(v)
            eigenvals.append(r)
    else:
        d = M.decomposition(algorithm='multimodular', proof=False)
        decomps = zip(fcp, [v[0] for v in d])
        for g, d in decomps:
            M_d = M.restrict(d, check=False)
            roots = [r[0] for r in g[0].roots(field)]
            
            # lift back to the original space
            ev = ws_ev(M_d, field) * d.basis_matrix()
            for r in roots:
                eigenvals.append(r)
                v = []
                for x in ev:
                    if isinstance(x, NumberFieldElement):
                        v.append(x.polynomial()(r))
                    else:
                        v.append(x)
                evecs.append(v)
                
    return [eigenvals, evecs]

def ws_ev(M, field, names='a'):
    t = M
    f = M.charpoly()
    roots = [r[0] for r in f.roots(field)]
    evs = []
    
    # Write down the eigenvector.
    # Write f(x) = (x-alpha)*g(x), where alpha is a root
    # of f(x).
    n = f.degree()
    if n > 1:
        R = f.parent()
        K = R.base_ring().extension(f, names=names)
        alpha = K.gen()
        beta = ~alpha   # multiplicative inverse of alpha
        c = [-f[0]*beta]
        for i in range(1,n-1):
            c.append((c[i-1] - f[i])*beta)
        c.append( K(1) )
    else:
        K = M.base_ring()
        c = [1]
    # The entries of c are the coefficients of g (as stated in
    # William Stein's Ph.D. thesis, Section 3.5.3).  We compute
    # g(t)v for a some vector v, and get an eigenvector.
    V = FreeModule(K, n)   
    t = t.change_ring(K)      # coerce t to be over K.
    for j in range(n):
        v = V.gen(j)
        I = t.iterates(v, n)  # iterates v, v*t, v*t^2, ...
        w = V(0)
        for i in range(n):
            w += c[i]*V(I.row(i).list())
        if w != 0:
            break
                    
    return w
