# Utility Functions and Classes for the PyInline.C module.

import re
from PyInline import BuildError

# these do not include return values
c_keywords = ["auto", "break", "case", "const", "continue", "default", "do", "else", "enum", "extern", "for", "goto", "if", "register", "return", "signed", "sizeof", "static", "struct", "switch", "typedef", "union", "unsigned", "volatile", "while"]

c_directive = '\s*#.*'
c_cppcomment = '//.*'
c_simplecomment = '/\*[^*]*\*+([^/*][^*]*\*+)*/'
c_doublequote = r'(?:"(?:\\.|[^"\\])*")'
c_singlequote = r'(?:\'(?:\\.|[^\'\\])*\')'
c_comment = re.compile("(%s|%s)|(?:%s|%s|%s)" % (c_doublequote,
                                                 c_singlequote,
                                                 c_cppcomment,
                                                 c_simplecomment,
                                                 c_directive))
internal = re.compile('\s*internal\w*')

const = re.compile('\s*const\s*')
extern = re.compile('\s*extern\s*')
volatile = re.compile('\s*volatile\s*')
star = re.compile('\s*\*\s*')
_c_pandn = "((?:(?:[\w*]+)\s+)+\**)(\w+)"
c_pandm = re.compile(_c_pandn)
_c_pandmain = "((?:(?:[\w*]+)\s+)+\**)main"
c_pandmain = re.compile(_c_pandn)
_c_function = _c_pandn + "\s*\(([^\)]*)\)"
c_function_def = re.compile("(?:%s|%s)|(%s)" % (c_doublequote,
                                                c_singlequote,
                                                _c_function + "\s*(?:\{|;)"))
_c_function_main = _c_pandmain + "\s*\(([^\)]*)\)"
c_main_function_def = re.compile("(?:%s|%s)|(%s)" % (c_doublequote,
                                                c_singlequote,
                                                _c_function_main + "\s*(?:\{|;)"))
c_main_function_def_noargs = re.compile("main\s*\(\s*(?:void)?\s*\)")
c_function_decl = re.compile(_c_function + "\s*;")

trimwhite = re.compile("\s*(.*)\s*")

def renameMainDef(code):
    results = c_main_function_def.findall(code)
    for match in results:
        if match[0]:
            returnType = trimWhite(match[1]);
            args = trimWhite(match[2])
            if args != '' and args != 'void' and not internal.search(returnType):
                raise BuildError("The function 'main' may not have arguments."); 
    # the application segfaults if main() is called, so we rename
    # it _PyInline_main but then pass it to python as main().
    # we also strip the arguments, since they do not make sense
    # in the inlined function.
    return c_main_function_def_noargs.sub("_PyInline_main()", code)    


def preProcess(code):
    return c_comment.sub(lambda(match): match.group(1) or "", code)

def findFunctionDefs(code):
    functionNames = set([])
    functionDefs = []
    for match in c_function_def.findall(code):
        if match[0]:
            # make sure we do not already have a function with the same name
            name = trimWhite(match[2])
            if name in functionNames:
                print "Ignoring duplicate function '" + name + "'."
                continue
            if name == "main":
                print "Ignoring function 'main'  This function will not be available from Python."
                continue
            functionNames.add(name)

            returnType = trimWhite(match[1]);
            # make sure that we are not mistaking something else as a keyword
            # or that the method is flagged not to have a wrapper for python.
            if not(returnType in c_keywords or internal.search(returnType)):
                # Remove extern from the return type
                if extern.search(returnType):
                    returnType = extern.sub(" ", returnType)

                # Remove volatile from the return type
                if volatile.search(returnType):
                    returnType = volatile.sub(" ", returnType)

                functionDefs.append({'return_type': returnType,
                                     'name': trimWhite(name),
                                     'rawparams': trimWhite(match[3])})
    return functionDefs


_wsLeft = re.compile("^\s*")
_wsRight = re.compile("\s*$")
def trimWhite(str):
    str = _wsLeft.sub("", str)
    str = _wsRight.sub("", str)
    return str
    
if __name__ == '__main__':
    x = """#include <stdio.h>
const char* foo = "long int x(int a) {";
long int barf(int a, char *b) {
  int x, y;
  int x[24];
}

long int *** fkfkfk(char * sdfkj, int a, char *b) {
  int x, y;
}

"""
    print findFunctionDefs(x)
                                    

