/*
PlanckLED_18a
ProfHuster
2018-01-28

This sketch is for a Teensy 3.2 board to run the photoelectric effect.
Major 2 - uses transconductance amp to read current
Version 2_18c - read loop of reading at contant DAC
Version 2_18b - allows set of DAc and read of ACS's
Version 2_18a - First shot. Based on SPCI sketch
 */
#include <ADC.h>
#include "myStrings.h"

// Instrument constants
#define DAC_PIN (A14) // The pin with the true DAC output
#define N_DAC_BITS (12) // resolution of DAC
#define DAC_MAX (1<<N_DAC_BITS) // 4096 = 2^12
#define DAC_WAIT_USEC (0) // Settling time for DAC in microseconds
#define VCC (3.3) // By default the DAC uses the 3.3 V regulator as a reference
#define N_ADC_BITS (12) // resolution of analog to digital converter
#define V0_PIN (A3) // For voltage = Rfeedback * Isample
#define V1_PIN (A1) // Reading voltage across sample
#define V2_PIN (A2) // Reading voltage across sample

// Define constant parameters for program
#define INPUT_SIZE 72 // Size of buffer to hold commands
#define RSENS0 (1000.) // 1 kOhm resistor
#define NSTEPS0 (17) // Default number of voltage steps to take
#define NAVG0 (1000) // Default number of readings to average at each step
#define N_MS_SLEEP0 (100) // Default milliseconds to sleep at each reading

char inBuffer[INPUT_SIZE]; // Place to input

const char idnStr[] = "PlanckLED_18a"; // ID string

const char helpStr[] = "\
Commands:\n\
*IDN? - prints out ID String\n\
*RST - resets to defaults\n\
CONF? - print configuration\n\
RSENS <float> - Set sensing resistor to numeric value\n\
RSENS? - Query (tell) the value of the sensing resistor\n\
NSTEPS <integer> - Set number of steps to value\n\
NSTEPS? - Query the number of steps\n\
NAVG <integer> - Set number of points to average\n\
NAVG? - Query the number of points to average\n\
SLEEP <integer> - number of milliseconds to sleep before each reading\n\
SLEEP? - Query the number of milliseconds to sleep before each reading\n\
DAC <integer> - set the analog output\n\
READ? - get one set of readings (averaged) without changing DAC\n\
RDLOOP? - get reads and outputs nAvg at constant DAC\n\
DATA? - Take data and return values\n\
HELP? - prints out this string\n";

/*
 * Parameters
 * These are user settable parameters
 */
float rSens = RSENS0; // Sensing resistance
int nSteps = NSTEPS0; // number of points to take
int nAvg = NAVG0; // number to average at each voltage
int nMsSleep = N_MS_SLEEP0; // milliseconds to sleep before each reading
int DACvalue = 0;

// ADC object
ADC *adc = new ADC(); // adc object;

// the setup function runs once when you press reset or power the board
void setup() {

  // Setup serial USB communication
  Serial.begin(250000);
  // wait for Serial USB connection
  while(!Serial)
    delay(1);

// When it starts, send the id string, the configuration, and the help string
  idn();
  conf();
  help(); // print out help string when the sketch first runs

// Set up the Teensy read and write pins
  analogWriteResolution(N_DAC_BITS);
  analogWrite(DAC_PIN, (int)0); // set to zero output
  pinMode(V0_PIN, INPUT);
  pinMode(V1_PIN, INPUT);
  pinMode(V2_PIN, INPUT);
  adc->setReference(ADC_REF_3V3);
  adc->setResolution(N_ADC_BITS);

}

// the loop function runs over and over again forever
void loop() {
  // Try to read a command. Just loop if no command
  while(!readSerialUntil(inBuffer, INPUT_SIZE, '\n'))
    delay(1); // Sleep 1 millisecond
  // convert input to upper case
  toUpperCase(inBuffer);
  // print command back to computer
  Serial.print(inBuffer);

  // Now check for each command. If found, call the command function
  if(found(inBuffer, "*RST")){
    rst();
  } else if(found(inBuffer, "*IDN?")){
    idn();
  } else if(found(inBuffer, "CONF?")){
    conf();
  } else if(found(inBuffer,"RSENS?")){
    rSensQuery();
  } else if(found(inBuffer,"RSENS")){
    setRSens();
  } else if(found(inBuffer,"NSTEPS?")){
    nStepsQuery();
  } else if(found(inBuffer,"NSTEPS")){
    setNSteps();
  } else if(found(inBuffer,"NAVG?")){
    nAvgQuery();
  } else if(found(inBuffer,"NAVG")){
    setNAvg();
  } else if(found(inBuffer,"SLEEP?")){
    nMsSleepQuery();
  } else if(found(inBuffer,"DAC")){
    setDAC();
  } else if(found(inBuffer,"READ?")){
    getReads();
  } else if(found(inBuffer,"RDLOOP?")){
    getRdLoop();
  } else if(found(inBuffer,"SLEEP")){
    setNMsSleep();
  } else if(found(inBuffer,"DATA?")){
    dataQuery();
  } else if(found(inBuffer,"HELP?")){
    help();
  } else {
    // If input does not make sense, print the help string
    err();
  }
  // Wait 1 second (== 1000 milliseconds) to loop again
  delay(1000);
}

// The SCPI commands are processed in the functions below
// *IDN?
void idn(){
  Serial.println(idnStr);
}

// *RST - Resets all parameters to initial state.
void rst(){
  rSens = RSENS0;
  nSteps = NSTEPS0;
  nAvg = NAVG0;
  nMsSleep = N_MS_SLEEP0;
}

// CONF? - returns all parameter values.
void conf(){
  rSensQuery();
  nStepsQuery();
  nAvgQuery();
  nMsSleepQuery();
}

// RSENS?
void rSensQuery(){
  Serial.print("rSens = ");
  Serial.println(rSens);
}

// RSENS <ff.ff> Set the sensing resistor value
void setRSens(){
  // Skip the "RSENS" part of the command string
  float value = atof(inBuffer+5);
  if(value >= 0.0){
    rSens = value;
  }
}

// NSTEPS?
void nStepsQuery(){
  Serial.print("nSteps = ");
  Serial.println(nSteps);
}

// NSTEPS <d> Set the number of steps
void setNSteps(){
  // Skip the "NSTEPS" part of the command string
  int value = atoi(inBuffer+6);
  if(value < 2)
    value = 2;
  else if(value > DAC_MAX)
    value = DAC_MAX;
  nSteps = value;
}

// NAVG? 
void nAvgQuery(){
  Serial.print("nAvg = ");
  Serial.println(nAvg);
}

// NAVG <d> Set the number of points to average
void setNAvg(){
  // Skip the "NAVG" part of the command string
  int value = atoi(inBuffer+4);
  if(value >= 0){
    nAvg = value;
  }
}

// SLEEP? 
void nMsSleepQuery(){
  Serial.print("nMsSleep = ");
  Serial.println(nMsSleep);
}

// SLEEP <d> Set the number of milliseconds to sleep before each reading
void setNMsSleep(){
  // Skip the "SLEEP" part of the command string
  int value = atoi(inBuffer+5);
  if(value >= 0){
    nMsSleep = value;
  }
}

// DAC <d> Set the DAC output
void setDAC(){
  // Skip the "DAC" part of the command string
  int value = atoi(inBuffer+3);
  DACvalue = value;
  if(value >= 0){
    analogWrite(DAC_PIN, value);
  }
}

// READ? Get one average reading without changing DAC
void getReads(){
  float adcMax = (float)(adc->getMaxValue(ADC_0));
  int intPin0=0, intPin0Sum=0;
  int intPin1=0, intPin1Sum=0;
  int intPin2=0, intPin2Sum=0;
  float VSample, ISample;

  // print header
  Serial.println("# V0  ,   V1  ,   V2  ,   Vs  , Is (mA)");
    // read voltages
    intPin0Sum = 0;
    intPin1Sum = 0;
    intPin2Sum = 0;
    // Loop to Average the Data
    for(int j=0; j < nAvg; j++){
      intPin0 = adc->analogRead(V0_PIN);
      intPin0Sum += intPin0;
      intPin1 = adc->analogRead(V1_PIN);      
      intPin1Sum += intPin1;
      intPin2 = adc->analogRead(V2_PIN);      
      intPin2Sum += intPin2;
    }
    float adcAvg0 = intPin0Sum / (float)nAvg;
    float adcAvg1 = intPin1Sum / (float)nAvg;
    float adcAvg2 = intPin2Sum / (float)nAvg;
    float V0 = adcAvg0 * VCC / adcMax;
    float V1 = adcAvg1 * VCC / adcMax;
    float V2 = adcAvg2 * VCC / adcMax;
    VSample = V2 - V1;
    ISample = 1e3 * (V1 - V0) / rSens;
    Serial.print(V0, 4);
    Serial.print(", ");
    Serial.print(V1, 4);
    Serial.print(", ");
    Serial.print(V2, 4);
    Serial.print(", ");
    Serial.print(VSample, 4);
    Serial.print(", ");
    Serial.print(ISample,4); // print current in milliamps
    Serial.println("");
}

// RDLOOP? Get whole set of navg of readings without changing DAC
void getRdLoop(){
  float adcMax = (float)(adc->getMaxValue(ADC_0));
  int intPin0=0, intPin0Sum=0;
  int intPin1=0, intPin1Sum=0;
  int intPin2=0, intPin2Sum=0;
  float VSample, ISample;

  // print header
  Serial.println("# V0  ,   V1  ,   V2  ,   Vs  , Is (mA)");
    // read voltages
    intPin0Sum = 0;
    intPin1Sum = 0;
    intPin2Sum = 0;
    // Loop to Average the Data
    for(int j=0; j < nAvg; j++){
      intPin0 = adc->analogRead(V0_PIN);
      intPin0Sum += intPin0;
      intPin1 = adc->analogRead(V1_PIN);      
      intPin1Sum += intPin1;
      intPin2 = adc->analogRead(V2_PIN);      
      intPin2Sum += intPin2;
      float V0 = intPin0 * VCC / adcMax;
      float V1 = intPin1 * VCC / adcMax;
      float V2 = intPin2 * VCC / adcMax;
      Serial.print(V0, 4);
      Serial.print(", ");
      Serial.print(V1, 4);
      Serial.print(", ");
      Serial.println(V2, 4);
     }
    float adcAvg0 = intPin0Sum / (float)nAvg;
    float adcAvg1 = intPin1Sum / (float)nAvg;
    float adcAvg2 = intPin2Sum / (float)nAvg;
    float V0 = adcAvg0 * VCC / adcMax;
    float V1 = adcAvg1 * VCC / adcMax;
    float V2 = adcAvg2 * VCC / adcMax;
    VSample = V2 - V1;
    ISample = 1e3 * (V1 - V0) / rSens;
    Serial.print(V0, 4);
    Serial.print(", ");
    Serial.print(V1, 4);
    Serial.print(", ");
    Serial.print(V2, 4);
    Serial.print(", ");
    Serial.print(VSample, 4);
    Serial.print(", ");
    Serial.print(ISample,4); // print current in milliamps
    Serial.println("");
}

/*
 * DATA?
 * The main function that sets voltage, reads the values and sends them 
 * to the USB Serial port
 */
void dataQuery(){
  float adcMax = (float)(adc->getMaxValue(ADC_0));
  int intPin0=0, intPin0Sum=0;
  int intPin1=0, intPin1Sum=0;
  int intPin2=0, intPin2Sum=0;
  float VSample, ISample;

  // print header
  Serial.println("# V0  ,   V1  ,   V2  ,   Vs  , Is (mA)");
  // Reset DAC Output
  analogWrite(DAC_PIN, (int)0);
  delayMicroseconds(DAC_WAIT_USEC);
  delay(nMsSleep);
// Main Data Loop
  for(int i = 0; i < nSteps; i++){
    // Set output voltage
    int DACOut = (i * DAC_MAX) / (nSteps - 1);
    digitalWrite(13, HIGH);
    analogWrite(DAC_PIN, DACOut);
    delayMicroseconds(DAC_WAIT_USEC);
    delay(nMsSleep);
    // read voltages
    intPin0Sum = 0;
    intPin1Sum = 0;
    intPin2Sum = 0;
    // Loop to Average the Data
    for(int j=0; j < nAvg; j++){
      intPin0 = adc->analogRead(V0_PIN);
      intPin0Sum += intPin0;
      intPin1 = adc->analogRead(V1_PIN);      
      intPin1Sum += intPin1;
      intPin2 = adc->analogRead(V2_PIN);      
      intPin2Sum += intPin2;
    }
    float adcAvg0 = intPin0Sum / (float)nAvg;
    float adcAvg1 = intPin1Sum / (float)nAvg;
    float adcAvg2 = intPin2Sum / (float)nAvg;
    float V0 = adcAvg0 * VCC / adcMax;
    float V1 = adcAvg1 * VCC / adcMax;
    float V2 = adcAvg2 * VCC / adcMax;
    VSample = V2 - V1;
    ISample = 1e3 * (V1 - V0) / rSens;
    Serial.print(V0, 4);
    Serial.print(", ");
    Serial.print(V1, 4);
    Serial.print(", ");
    Serial.print(V2, 4);
    Serial.print(", ");
    Serial.print(VSample, 4);
    Serial.print(", ");
    Serial.print(ISample,4); // print current in milliamps
    Serial.println("");
  }
  analogWrite(DAC_PIN, (int)0); // set to zero output at end
  Serial.println("# Done");
  digitalWrite(13, LOW);
  DACvalue = 0;
}

// HELP?
void help(){
  Serial.print(helpStr);
}
// Error
void err(){
  Serial.println("ERROR");
}
// EOF
